

#include <PalmOS.h>
#include <PalmCompatibility.h>
#include <Handspring/HsExt.h>

#include <stdarg.h>

#include "springboard.h"
#include "format.h"







void _putchar(int c, void *cookie)
{
	char **ptr = (char **)cookie;
	**ptr = c;
	(*ptr)++;
}

int sprintf(char *str, const char *form, ...)
{
	int i;
	va_list ap;
	va_start(ap, form);
	i = __vformat(_putchar, &str, form, ap);
	va_end(ap);
	*str = 0;
	return i;
}





template <typename T> class PalmObject {
protected:
	typedef T Type;
	T _object;
public:
	PalmObject()
	{
		Attach(NULL);
	}
	PalmObject(Type object)
	{
		Attach(object);
	}
	void Attach(Type field)
	{
		_object = field;
	}
	Type Detach()
	{
		Type field = _object;
		_object = NULL;
		return field;
	}
	operator Type ()
	{
		return _object;
	}
};

class PalmForm : public PalmObject<FormType *> {
public:
	PalmForm()
		: PalmObject<FormType *>()
	{
	}
	PalmForm(Type t)
		: PalmObject<FormType *>(t)
	{
	}

	static void DrawActiveForm()
	{
		FrmDrawForm(FrmGetActiveForm());
	}

	void *GetObject(UInt16 objID)
	{
		return GetObjectPtr(GetObjectIndex(objID));
	}



	inline FormType *InitForm (UInt16 rscID)
	{
		Attach(FrmInitForm(rscID));
	}
	inline void DeleteForm()
	{
		FrmDeleteForm(_object);
	}
	inline void DrawForm()
	{
		FrmDrawForm(_object);
	}
	inline void EraseForm()
	{
		FrmEraseForm(_object);
	}
	inline FormType *GetActiveForm()
	{
		Attach(FrmGetActiveForm());
	}
	inline void SetActiveForm()
	{
		FrmSetActiveForm(_object);
	}
	inline UInt16 GetActiveFormID()
	{
		return FrmGetActiveFormID();
	}
	inline Boolean GetUserModifiedState()
	{
		return FrmGetUserModifiedState(_object);
	}
	inline void SetNotUserModified()
	{
		FrmSetNotUserModified(_object);
	}
	inline UInt16 GetFocus()
	{
		return FrmGetFocus(_object);
	}
	inline void SetFocus(UInt16 fieldIndex)
	{
		FrmSetFocus(_object, fieldIndex);
	}
	inline Boolean HandleEvent(EventType *eventP)
	{
		return FrmHandleEvent(_object, eventP);
	}
	inline void GetFormBounds(RectangleType *rP)
	{
		FrmGetFormBounds(_object, rP);
	}
	inline WinHandle GetWindowHandle()
	{
		return FrmGetWindowHandle(_object);
	}
	inline UInt16 GetFormId()
	{
		return FrmGetFormId(_object);
	}
	inline FormType *GetFormPtr(UInt16 formId)
	{
		Attach(FrmGetFormPtr(formId));
	}
	inline FormType *GetFirstForm()
	{
		Attach(FrmGetFirstForm());
	}
	inline UInt16 GetNumberOfObjects()
	{
		return FrmGetNumberOfObjects(_object);
	}
	inline UInt16 GetObjectIndex(UInt16 objID)
	{
		return FrmGetObjectIndex(_object, objID);
	}
	inline UInt16 GetObjectId(UInt16 objIndex)
	{
		return FrmGetObjectId(_object, objIndex);
	}
	inline FormObjectKind GetObjectType(UInt16 objIndex)
	{
		return FrmGetObjectType(_object, objIndex);
	}
	inline void *GetObjectPtr(UInt16 objIndex)
	{
		return FrmGetObjectPtr(_object, objIndex);
	}
	inline void GetObjectBounds(UInt16 objIndex, RectangleType *rP)
	{
		FrmGetObjectBounds(_object, objIndex, rP);
	}
	inline void HideObject(UInt16 objIndex)
	{
		FrmHideObject(_object, objIndex);
	}
	inline void ShowObject(UInt16 objIndex)
	{
		FrmShowObject(_object, objIndex);
	}
	inline void GetObjectPosition(UInt16 objIndex, Coord *x, Coord *y)
	{
		FrmGetObjectPosition(_object, objIndex, x, y);
	}
	inline void SetObjectPosition(UInt16 objIndex, Coord x, Coord y)
	{
		FrmSetObjectPosition(_object, objIndex, x, y);
	}
	inline void SetObjectBounds(UInt16 objIndex, const RectangleType *bounds)
	{
		FrmSetObjectBounds(_object, objIndex, bounds);
	}
	inline Int16 GetControlValue(UInt16 controlID)
	{
		return FrmGetControlValue(_object, controlID);
	}
	inline void SetControlValue(UInt16 controlID, Int16 newValue)
	{
		FrmSetControlValue(_object, controlID, newValue);
	}
	inline UInt16 GetControlGroupSelection(UInt8 groupNum)
	{
		return FrmGetControlGroupSelection(_object, groupNum);
	}
	inline void SetControlGroupSelection(UInt8 groupNum, UInt16 controlID)
	{
		FrmSetControlGroupSelection(_object, groupNum, controlID);
	}
	inline void CopyLabel(UInt16 labelID, const Char *newLabel)
	{
		FrmCopyLabel(_object, labelID, newLabel);
	}
	inline const Char *GetLabel(UInt16 labelID)
	{
		return FrmGetLabel(_object, labelID);
	}
	inline void SetCategoryLabel(UInt16 objIndex, Char *newLabel)
	{
		FrmSetCategoryLabel(_object, objIndex, newLabel);
	}
	inline const Char *GetTitle()
	{
		return FrmGetTitle(_object);
	}
	inline void SetTitle(Char *newTitle)
	{
		FrmSetTitle(_object, newTitle);
	}
	inline void CopyTitle(const Char *newTitle)
	{
		FrmCopyTitle(_object, newTitle);
	}
	inline void *GetGadgetData(UInt16 objIndex)
	{
		return FrmGetGadgetData(_object, objIndex);
	}
	inline void SetGadgetData(UInt16 objIndex, const void *data)
	{
		FrmSetGadgetData(_object, objIndex, data);
	}
	inline void SetGadgetHandler(UInt16 objIndex, FormGadgetHandlerType *attrP)
	{
		FrmSetGadgetHandler(_object, objIndex, attrP);
	}
	inline UInt16 DoDialog()
	{
		return FrmDoDialog(_object);
	}
	inline UInt16 Alert(UInt16 alertId)
	{
		return FrmAlert(alertId);
	}
	inline UInt16 CustomAlert(UInt16 alertId, const Char *s1, const Char *s2, const Char *s3)
	{
		return FrmCustomAlert(alertId, s1, s2, s3);
	}
	inline void Help(UInt16 helpMsgId)
	{
		FrmHelp(helpMsgId);
	}
	inline void UpdateScrollers(UInt16 upIndex, UInt16 downIndex, Boolean scrollableUp, Boolean scrollableDown)
	{
		FrmUpdateScrollers(_object, upIndex, downIndex, scrollableUp, scrollableDown);
	}
	inline Boolean Visible()
	{
		return FrmVisible(_object);
	}
	inline void SetEventHandler(FormEventHandlerType *handler)
	{
		FrmSetEventHandler(_object, handler);
	}
	inline Boolean DispatchEvent(EventType *eventP)
	{
		return FrmDispatchEvent(eventP);
	}

	static void PopupForm(UInt16 formId)
	{
		FrmPopupForm(formId);
	}
	static void GotoForm(UInt16 formId)
	{
		FrmGotoForm(formId);
	}
	static void UpdateForm(UInt16 formId, UInt16 updateCode)
	{
		FrmUpdateForm(formId, updateCode);
	}
	static void ReturnToForm(UInt16 formId)
	{
		FrmReturnToForm(formId);
	}
	static void CloseAllForms()
	{
		FrmCloseAllForms();
	}
	static void SaveAllForms()
	{
		FrmSaveAllForms();
	}

	inline Boolean PointInTitle(Coord x, Coord y)
	{
		return FrmPointInTitle(_object, x, y);
	}
	inline void SetMenu(UInt16 menuRscID)
	{
		FrmSetMenu(_object, menuRscID);
	}
	inline Boolean ValidatePtr()
	{
		return FrmValidatePtr(_object);
	}

	// extern Err FrmAddSpaceForObject(FormType **formPP, MemPtr *objectPP, FormObjectKind objectKind, UInt16 objectSize)
	// extern Err FrmRemoveObject(FormType **formPP, UInt16 objIndex)
	// extern FormType *FrmNewForm(UInt16 formID, const Char *titleStrP, Coord x, Coord y, Coord width, Coord height, Boolean modal, UInt16 defaultButton, UInt16 helpRscID, UInt16 menuRscID)
	// extern FormLabelType *FrmNewLabel(FormType **formPP, UInt16 ID, const Char *textP, Coord x, Coord y, FontID font)
	// extern FormBitmapType *FrmNewBitmap(FormType **formPP, UInt16 ID, UInt16 rscID, Coord x, Coord y)
	// extern FormGadgetType *FrmNewGadget(FormType **formPP, UInt16 id, Coord x, Coord y, Coord width, Coord height)
	// extern Err FrmActiveState(FormActiveStateType *stateP, Boolean save)
	// extern UInt16 FrmCustomResponseAlert(UInt16 alertId, const Char *s1, const Char *s2, const Char *s3, Char *entryStringBuf, Int16 entryStringBufLength, FormCheckResponseFuncPtr callback)
	// extern FrmGraffitiStateType *FrmNewGsi(FormType **formPP, Coord x, Coord y)
	// #define FrmSaveActiveState(stateP)			FrmActiveState(stateP, true)
	// #define FrmRestoreActiveState(stateP)		FrmActiveState(stateP, false)
};

class PalmActiveForm : public PalmForm {
public:
	PalmActiveForm()
	{
		GetActiveForm();
	}

};

class PalmControl : public PalmObject<ControlType *> {
public:
	PalmControl()
		: PalmObject<ControlType *>()
	{
	}
	PalmControl(Type t)
		: PalmObject<ControlType *>(t)
	{
	}
	inline void DrawControl()
	{
		CtlDrawControl(_object);
	}
	inline void EraseControl()
	{
		CtlEraseControl(_object);
	}
	inline void HideControl()
	{
		CtlHideControl(_object);
	}
	inline void ShowControl()
	{
		CtlShowControl(_object);
	}
	inline bool Enabled()
	{
		return CtlEnabled(_object);
	}
	inline void SetEnabled(bool usable)
	{
		CtlSetEnabled(_object, usable);
	}
	inline void SetUsable(bool usable)
	{
		CtlSetUsable(_object, usable);
	}
	inline Int16 GetValue()
	{
		return CtlGetValue(_object);
	}
	inline void SetValue(Int16 newValue)
	{
		CtlSetValue(_object, newValue);
	}
	inline const Char *GetLabel()
	{
		return CtlGetLabel(_object);
	}
	inline void SetLabel(const Char *newLabel)
	{
		CtlSetLabel(_object, newLabel);
	}
	inline void SetGraphics(ControlType *ctlP, DmResID newBitmapID, DmResID newSelectedBitmapID)
	{
		CtlSetGraphics(ctlP, newBitmapID, newSelectedBitmapID);
	}
	inline void SetSliderValues(ControlType *ctlP, const UInt16 *minValueP, const UInt16 *maxValueP, const UInt16 *pageSizeP, const UInt16 *valueP)
	{
		CtlSetSliderValues(ctlP, minValueP, maxValueP, pageSizeP, valueP);
	}
	inline void GetSliderValues(const ControlType *ctlP, UInt16 *minValueP, UInt16 *maxValueP, UInt16 *pageSizeP, UInt16 *valueP)
	{
		CtlGetSliderValues(ctlP, minValueP, maxValueP, pageSizeP, valueP);
	}
	inline void HitControl()
	{
		CtlHitControl(_object);
	}
	inline bool HandleEvent(EventType *pEvent)
	{
		return CtlHandleEvent(_object, pEvent);
	}
	inline bool ValidatePointer()
	{
		return CtlValidatePointer(_object);
	}
};

class PalmField : public PalmObject<FieldType *> {
public:
	PalmField()
		: PalmObject<FieldType *>()
	{
	}
	PalmField(Type t)
		: PalmObject<FieldType *>(t)
	{
	}

	inline void Copy()
	{
		FldCopy(_object);
	}
	inline void Cut()
	{
		FldCut(_object);
	}
	inline void DrawField()
	{
		FldDrawField(_object);
	}
	inline void EraseField()
	{
		FldEraseField(_object);
	}
	inline void FreeMemory()
	{
		FldFreeMemory(_object);
	}
	inline void GetBounds(RectanglePtr rect)
	{
		FldGetBounds(_object, rect);
	}
	inline FontID GetFont()
	{
		return FldGetFont(_object);
	}
	inline void GetSelection(UInt16 *startPosition, UInt16 *endPosition)
	{
		FldGetSelection(_object, startPosition, endPosition);
	}
	inline MemHandle GetTextHandle()
	{
		return FldGetTextHandle(_object);
	}
	inline Char *GetTextPtr()
	{
		return FldGetTextPtr(_object);
	}
	inline Boolean HandleEvent(EventType *eventP)
	{
		return FldHandleEvent(_object, eventP);
	}
	inline void Paste()
	{
		FldPaste(_object);
	}
	inline void RecalculateField(Boolean redraw)
	{
		FldRecalculateField(_object, redraw);
	}
	inline void SetBounds(const RectangleType *rP)
	{
		FldSetBounds(_object, rP);
	}
	inline void SetFont(FontID fontID)
	{
		FldSetFont(_object, fontID);
	}
	inline void SetText(MemHandle textHandle, UInt16 offset, UInt16 size)
	{
		FldSetText(_object, textHandle, offset, size);
	}
	inline void SetTextHandle(MemHandle textHandle)
	{
		FldSetTextHandle(_object, textHandle);
	}
	inline void SetTextPtr(Char *textP)
	{
		FldSetTextPtr(_object, textP);
	}
	inline void SetUsable(Boolean usable)
	{
		FldSetUsable(_object, usable);
	}
	inline void SetSelection(UInt16 startPosition, UInt16 endPosition)
	{
		FldSetSelection(_object, startPosition, endPosition);
	}
	inline void GrabFocus()
	{
		FldGrabFocus(_object);
	}
	inline void ReleaseFocus()
	{
		FldReleaseFocus(_object);
	}
	inline UInt16 GetInsPtPosition()
	{
		return FldGetInsPtPosition(_object);
	}
	inline void SetInsPtPosition(UInt16 pos)
	{
		FldSetInsPtPosition(_object, pos);
	}
	inline void SetInsertionPoint(UInt16 pos)
	{
		FldSetInsertionPoint(_object, pos);
	}
	inline UInt16 GetScrollPosition()
	{
		return FldGetScrollPosition(_object);
	}
	inline void SetScrollPosition(UInt16 pos)
	{
		FldSetScrollPosition(_object, pos);
	}
	inline void GetScrollValues(UInt16 *scrollPosP, UInt16 *textHeightP, UInt16 *fieldHeightP)
	{
		FldGetScrollValues(_object, scrollPosP, textHeightP, fieldHeightP);
	}
	inline UInt16 GetTextLength()
	{
		return FldGetTextLength(_object);
	}
	inline void ScrollField(UInt16 linesToScroll, WinDirectionType direction)
	{
		FldScrollField(_object, linesToScroll, direction);
	}
	inline Boolean Scrollable(WinDirectionType direction)
	{
		return FldScrollable(_object, direction);
	}
	inline UInt16 GetVisibleLines()
	{
		return FldGetVisibleLines(_object);
	}
	inline UInt16 GetTextHeight()
	{
		return FldGetTextHeight(_object);
	}
	inline static UInt16 CalcFieldHeight(const Char *chars, UInt16 maxWidth)
	{
		return FldCalcFieldHeight(chars, maxWidth);
	}
	inline static UInt16 WordWrap(const Char *chars, Int16 maxWidth)
	{
		return FldWordWrap(chars, maxWidth);
	}
	inline void CompactText()
	{
		FldCompactText(_object);
	}
	inline Boolean Dirty()
	{
		return FldDirty(_object);
	}
	inline void SetDirty(Boolean dirty)
	{
		FldSetDirty(_object, dirty);
	}
	inline UInt16 GetMaxChars()
	{
		return FldGetMaxChars(_object);
	}
	inline void SetMaxChars(UInt16 maxChars)
	{
		FldSetMaxChars(_object, maxChars);
	}
	inline Boolean Insert(const Char *insertChars, UInt16 insertLen)
	{
		FldInsert(_object, insertChars, insertLen);
	}
	inline void Delete(UInt16 start, UInt16 end)
	{
		FldDelete(_object, start, end);
	}
	inline void Undo()
	{
		FldUndo(_object);
	}
	inline UInt16 GetTextAllocatedSize()
	{
		return FldGetTextAllocatedSize(_object);
	}
	inline void SetTextAllocatedSize(UInt16 allocatedSize)
	{
		FldSetTextAllocatedSize(_object, allocatedSize);
	}
	inline void GetAttributes(FieldAttrPtr attrP)
	{
		FldGetAttributes(_object, attrP);
	}
	inline void SetAttributes(const FieldAttrType *attrP)
	{
		FldSetAttributes(_object, attrP);
	}
	inline void SendChangeNotification()
	{
		FldSendChangeNotification(_object);
	}
	inline void SendHeightChangeNotification(UInt16 pos, Int16 numLines)
	{
		FldSendHeightChangeNotification(_object, pos, numLines);
	}
	inline Boolean MakeFullyVisible()
	{
		return FldMakeFullyVisible(_object);
	}
	inline UInt16 GetNumberOfBlankLines()
	{
		return FldGetNumberOfBlankLines(_object);
	}
	inline static FieldType *NewField(void **formPP, UInt16 id, Coord x, Coord y, Coord width, Coord height, FontID font, UInt32 maxChars, Boolean editable, Boolean underlined, Boolean singleLine, Boolean dynamicSize, JustificationType justification, Boolean autoShift, Boolean hasScrollBar, Boolean numeric)
	{
		return FldNewField(formPP, id, x, y, width, height, font, maxChars, editable, underlined, singleLine, dynamicSize, justification, autoShift, hasScrollBar, numeric);
	}
};

class PalmDatabase : public PalmObject<DmOpenRef> {
public:
	bool OpenDatabaseByTypeCreator(UInt32 type, UInt32 creator, UInt16 mode)
	{
		Attach(DmOpenDatabaseByTypeCreator(type, creator, mode));
		return _object != NULL;
	}
	MemHandle NewRecord(UInt16 *atP, UInt32 size)
	{
		return DmNewRecord(_object, atP, size);
	}
	Err ReleaseRecord(UInt16 index, bool dirty)
	{
		return DmReleaseRecord(_object, index, dirty);
	}
};

class PalmApplication {
public:
	PalmApplication()
	{
	}
	virtual ~PalmApplication()
	{
	}
	bool Startup();
	void Cleanup();
	bool Run();

protected:
	bool FormEventHandler(EventType *event);
	bool OpenDatabase(const char *name, UInt32 creator, UInt32 type);
	bool FormEventHandle(EventType *event);

	virtual bool OnCommand(int command) {}

	virtual bool OnFormLoad(EventType *event);
	virtual bool OnNil(EventType *event) {}
	virtual bool OnFormOpen(EventType *event) {}
	virtual bool OnMenu(EventType *event) {}
	virtual bool OnControlSelect(EventType *event) {}

private:
	PalmDatabase _database;

	static Boolean _FormEventHandle(EventType *event);
};







bool PalmApplication::FormEventHandle(EventType *event)
{
	switch (event->eType) {
	case frmOpenEvent:		return OnFormOpen(event);
	case ctlSelectEvent:	return OnControlSelect(event);
	case menuEvent:			return OnMenu(event);
	case nilEvent:			return OnNil(event);
	}
	return false;
}



bool PalmApplication::OpenDatabase(const char *name, UInt32 creator, UInt32 type)
{
	void *p;
	VoidHand h;
	UInt index;

	if (!_database.OpenDatabaseByTypeCreator(type, creator, dmModeReadWrite)) {
		if (DmCreateDatabase(0, name, creator, type, false))
			return false;

		_database.OpenDatabaseByTypeCreator(type, creator, dmModeReadWrite);

		index = 0;
		h = _database.NewRecord(&index, 1);
		p = MemHandleLock(h);

		DmWrite(p, 0, "", 1);
		MemPtrUnlock(p);

		_database.ReleaseRecord(index, true);
	}
	return true;
}


UInt32 SupportDepth;
Boolean SupportColor_flag;
UInt32 OldDepth;


bool PalmApplication::Startup()
{
	if (!OpenDatabase("SpringboardTestDB", 'SpBd', 'Data')) return false;

	//

	UInt32 NewDepth;

	FtrGet(sysFtrCreator,sysFtrNumDisplayDepth,&OldDepth);
	ScrDisplayMode(scrDisplayModeGetSupportedDepths,NULL,NULL,&SupportDepth,NULL);
	ScrDisplayMode(scrDisplayModeGetSupportsColor,NULL,NULL,NULL,&SupportColor_flag);

	if      (SupportDepth & 0x0080) NewDepth = 8;
	else if (SupportDepth & 0x0008) NewDepth = 4;
	else if (SupportDepth & 0x0002) NewDepth = 2;
	else                            NewDepth = 1;
	
	if (!FtrSet(sysFtrCreator, sysFtrNumDisplayDepth, NewDepth)) {
		ScrDisplayMode(scrDisplayModeSetToDefaults,NULL,NULL,NULL,NULL);
	}

	//
	
	FrmGotoForm(ID_FORM);
	return true;
}

void PalmApplication::Cleanup()
{
	FtrSet(sysFtrCreator,sysFtrNumDisplayDepth,OldDepth);
	ScrDisplayMode(scrDisplayModeSetToDefaults,NULL,NULL,NULL,NULL);

	//

	if ((DmOpenRef)_database) {
		DmReleaseRecord(_database, 0, true);
		DmCloseDatabase(_database);
	}
	
}






bool PalmApplication::OnFormLoad(EventType *event)
{
	int formID = event->data.frmLoad.formID;
	FormType *form = FrmInitForm(formID);
	FrmSetActiveForm(form);

	switch (formID) {
	case ID_FORM:
		FrmSetEventHandler(form, _FormEventHandle);
		break;
	}
	return true;
}

bool PalmApplication::Run()
{
	unsigned short err;
	EventType event;

	if (!Startup()) return false;

	do {

		EvtGetEvent(&event, sysTicksPerSecond / 10);

		if (SysHandleEvent(&event))
			continue;
		if (MenuHandleEvent(0, &event, &err))
			continue;

		if (event.eType == frmLoadEvent) {
			OnFormLoad(&event);
		}

		FrmDispatchEvent(&event);

	} while (event.eType != appStopEvent);

	Cleanup();
	return true;
}












class PalmTestApp : public PalmApplication {
public:
	PalmTestApp();
	virtual ~PalmTestApp();
private:
	int WhichBitmap;
	unsigned char _led;
	unsigned short *_base;
	unsigned long _size;
private:
	void SetLED(unsigned char value);
	bool GetLED(int index);
	void SetLED(int index, bool value);
protected:
	bool OnCommand(int command);
	virtual bool OnNil(EventType *event);
	virtual bool OnFormOpen(EventType *event);
	virtual bool OnMenu(EventType *event);
	virtual bool OnControlSelect(EventType *event);
};

PalmTestApp _app;


PalmTestApp::PalmTestApp()
{
	_led = 0;
	_base = 0;
	_size = 0;
	
	unsigned long base = 0;
	HsCardAttrGet(1, hsCardAttrCsBase, &base);
	if (base) {
		_base = (unsigned short *)base;
		HsCardAttrGet(1, hsCardAttrCsSize, &_size);
		SetLED(_led);
	}
}
PalmTestApp::~PalmTestApp()
{
		SetLED(0);
}

void PalmTestApp::SetLED(unsigned char value)
{
	if (_base) *_base = ~value;
}

bool PalmTestApp::GetLED(int index)
{
	return (_led >> index) & 1;
}

void PalmTestApp::SetLED(int index, bool value)
{
	if (!_base) return;

	index &= 7;

	int n = 1 << index;
	_led |= n;
	if (!value) _led ^= n;
	SetLED(_led);

	PalmActiveForm form;
	PalmControl chkbox((ControlType *)form.GetObject(ID_CHECK_BIT_0 + index));
	chkbox.SetValue(value);
}


Boolean PalmApplication::_FormEventHandle(EventType *event)
{
	return _app.FormEventHandle(event);
}


#include <stdlib.h>


void memset(void *p, unsigned char c, unsigned long n)
{
	unsigned char *q = (unsigned char *)p;
	while (n > 0) {
		*q++ = c;
		n--;
	}
}


struct ColorTable256 {
	UInt16 count;
	RGBColorType table[256];
};



struct PixelRGBA {
	unsigned char r, g, b, a;
};

bool PalmTestApp::OnCommand(int command)
{
	switch (command) {
	case ID_TEST:
		break;
	}
	return false;
}


bool PalmTestApp::OnNil(EventType *event)
{
#if 0
	static int _cycle = 0;
	static const char table[] = { 0, 1, 2, 3, 4, 5, 6, 7, 6, 5, 4, 3, 2, 1 };
	SetLED(table[_cycle % 14], false);
	_cycle = (_cycle + 1) % 14;
	SetLED(table[_cycle], true);
#endif
	return true;
}

bool PalmTestApp::OnFormOpen(EventType *event)
{
	PalmActiveForm form;
	form.DrawForm();
	return true;
}

bool PalmTestApp::OnMenu(EventType *event)
{
	int n, id;

	id = event->data.menu.itemID;

	n = 0;
	switch (id) {
	case ID_CHECK_BIT_7:	n++;
	case ID_CHECK_BIT_6:	n++;
	case ID_CHECK_BIT_5:	n++;
	case ID_CHECK_BIT_4:	n++;
	case ID_CHECK_BIT_3:	n++;
	case ID_CHECK_BIT_2:	n++;
	case ID_CHECK_BIT_1:	n++;
	case ID_CHECK_BIT_0:
		SetLED(n, !GetLED(n));
		break;

	case ID_ABOUT:
		FrmAlert(ID_ALERT_ABOUT);
		break;

	default:
		return OnCommand(event->data.menu.itemID);
	}
	return true;
}


bool PalmTestApp::OnControlSelect(EventType *event)
{
	int n, id;

	id = event->data.ctlSelect.controlID;

	n = 0;
	switch (id) {
	case ID_CHECK_BIT_7:	n++;
	case ID_CHECK_BIT_6:	n++;
	case ID_CHECK_BIT_5:	n++;
	case ID_CHECK_BIT_4:	n++;
	case ID_CHECK_BIT_3:	n++;
	case ID_CHECK_BIT_2:	n++;
	case ID_CHECK_BIT_1:	n++;
	case ID_CHECK_BIT_0:
		SetLED(n, event->data.ctlSelect.on);
		break;
	default:
		return OnCommand(id);
	}
	return false;
}


struct MyGlobalsType {
	int cardNo;
};

DWord PilotMain(Word cmd, Ptr cmdPBP, Word launchFlags)
{
	int error;
	unsigned long base, size;

	Err err;
	VoidHand codeResH;
	MyGlobalsType *globalsP;

	if (cmd == sysAppLaunchCmdNormalLaunch) {
		_app.PalmApplication::Run();
	}

	return 0;
}


