
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include "format.h"

#define INHIBIT_FLOAT


#ifndef INHIBIT_FLOAT
#include <math.h>
#endif

struct FormatInfo {
	int padding;
	int maxwidth;
	int precision;
	int leftalign;
	void (*output)(int c, void *cookie);
	void *cookie;
};

#define OUTPUT(info, c) info->output(c, info->cookie)

static void __output_s(const struct FormatInfo *info, const char *value)
{
	int i, n;

	n = strlen(value);
	if (!info->leftalign) {
		for (i = n; i < info->maxwidth; i++) {
			OUTPUT(info, info->padding);
		}
	}
	for (i = 0; i < n; i++) {
		OUTPUT(info, value[i]);
	}
	if (info->leftalign) {
		for (i = n; i < info->maxwidth; i++) {
			OUTPUT(info, info->padding);
		}
	}
}

static void __output_d(const struct FormatInfo *info, int value)
{
	int c, i, len, padlen, padchr, sign;
	char tmp[100];

	len = 0;
	sign = 0;
	if (value || info->precision) {
		if (value < 0) {
			sign = '-';
			value = -value;
		}
		do {
			tmp[len] = value % 10;
			value /= 10;
			len++;
		} while (value > 0 && len < sizeof(tmp));
	}

	padlen = 0;
	i = len;
	if (info->precision > 0 && i < info->precision) {
		i = info->precision;
	}
	if (sign) i++;
	if (i < info->maxwidth) {
		padlen = info->maxwidth - i;
	}
	padchr = info->precision < 0 ? info->padding : ' ';

	if (padlen > 0 && !info->leftalign) {
		if (sign && padchr == '0')
			OUTPUT(info, sign);
		while (padlen-- > 0)
			OUTPUT(info, padchr);
		if (sign && padchr != '0')
			OUTPUT(info, sign);
	} else if (sign) {
		OUTPUT(info, sign);
	}
	for (i = len; i < info->precision; i++) {
		OUTPUT(info, '0');
	}
	for (i = len; i > 0; i--) {
		OUTPUT(info, tmp[i - 1] + '0');
	}
	if (padlen > 0 && info->leftalign) {
		while (padlen-- > 0)
			OUTPUT(info, padchr);
	}
}

static void __output_u(const struct FormatInfo *info, unsigned long value)
{
	int c, i, len, padlen, padchr;
	char tmp[100];

	len = 0;
	if (value || info->precision) {
		do {
			tmp[len] = value % 10;
			value /= 10;
			len++;
		} while (value > 0 && len < sizeof(tmp));
	}

	padlen = 0;
	i = len;
	if (info->precision > 0 && i < info->precision) {
		i = info->precision;
	}
	if (i < info->maxwidth) {
		padlen = info->maxwidth - i;
	}
	padchr = info->leftalign < 0 ? ' ' : info->padding;

	if (padlen > 0 && !info->leftalign) {
		while (padlen-- > 0)
			OUTPUT(info, padchr);
	}
	for (i = len; i < info->precision; i++) {
		OUTPUT(info, '0');
	}
	for (i = len; i > 0; i--) {
		OUTPUT(info, tmp[i - 1] + '0');
	}
	if (padlen > 0 && info->leftalign) {
		while (padlen-- > 0)
			OUTPUT(info, padchr);
	}
}

static void __output_x(const struct FormatInfo *info, unsigned long value)
{
	int c, i, len, padlen, padchr;
	char tmp[100];

	len = 0;
	if (value || info->precision) {
		do {
			tmp[len] = value % 16;
			value /= 16;
			len++;
		} while (value > 0 && len < sizeof(tmp));
	}

	padlen = 0;
	i = len;
	if (info->precision > 0 && i < info->precision) {
		i = info->precision;
	}
	if (i < info->maxwidth) {
		padlen = info->maxwidth - i;
	}
	padchr = info->leftalign < 0 ? ' ' : info->padding;

	if (padlen > 0 && !info->leftalign) {
		while (padlen-- > 0)
			OUTPUT(info, padchr);
	}
	for (i = len; i < info->precision; i++) {
		OUTPUT(info, '0');
	}
	for (i = len; i > 0; i--) {
		OUTPUT(info, "0123456789abcdef"[tmp[i - 1]]);
	}
	if (padlen > 0 && info->leftalign) {
		while (padlen-- > 0)
			OUTPUT(info, padchr);
	}
}

#ifndef INHIBIT_FLOAT
static void __output_f(const struct FormatInfo *info, double value)
{
	int i, len, sign, padlen, padchr, precision;
	char tmp[100];

	precision = info->precision;
	if (precision < 0) precision = 6;

	sign = 0;
	if (value < 0) {
		sign = '-';
		value = -value;
	}
	for (i = 0; i < precision; i++) {
		value *= 10;
	}
	value = floor(value + 0.5);

	len = 0;
	while (value > 0 && len < sizeof(tmp)) {
		tmp[len] = (int)fmod(value, 10);
		value = floor(value / 10);
		len++;
	}
	if (len == precision && len < sizeof(tmp))
		tmp[len++] = 0;

	padlen = 0;
	i = len;
	if (sign) i++;
	if (precision) i++;
	if (i < info->maxwidth) {
		padlen = info->maxwidth - i;
	}
	padchr = info->leftalign < 0 ? ' ' : info->padding;


	if (!info->leftalign) {
		while (padlen-- > 0)
			OUTPUT(info, padchr);
	}
	if (sign) {
		OUTPUT(info, sign);
	}
	for (i = len; i > precision; i--) {
		OUTPUT(info, tmp[i - 1] + '0');
	}
	if (precision > 0) {
		OUTPUT(info, '.');
		for (i = precision; i > 0; i--) {
			OUTPUT(info, tmp[i - 1] + '0');
		}
	}
	if (info->leftalign) {
		while (padlen-- > 0)
			OUTPUT(info, padchr);
	}
}
#endif

int __vformat(void (*output)(int c, void *cookie), void *cookie, const char *format, va_list ap)
{
	int c, n, mode;
	unsigned long u;
	struct FormatInfo info;
	info.output = output;
	info.cookie = cookie;
	mode = 0;
	while (*format) {
		c = *format++;
		switch (mode) {
		case '%':
			if (c == '.') {
				if (info.precision < 0) {
					if (info.maxwidth < 0) info.maxwidth = 0;
					info.precision = 0;
				}
			} else if (c == '*') {
				u = va_arg(ap, unsigned long);
				if (info.precision < 0) {
					info.maxwidth = u;
				} else {
					info.precision = u;
				}
			} else if (c == '-') {
				info.leftalign = 1;
			} else if (isdigit(c)) {
				n = c - '0';
				if (info.maxwidth < 0) {
					info.maxwidth = n;
					if (c == '0') info.padding = '0';
				} else if (info.precision < 0) {
					info.maxwidth = info.maxwidth * 10 + n;				
				} else {
					info.precision = info.precision * 10 + n;
				}
			} else {
				if (info.leftalign && info.padding == '0') {
					info.padding = ' ';
				}
				switch (c) {
				case '%':
					output('%', cookie);
					break;
				case 'c':
					output(va_arg(ap, int), cookie);
					break;
				case 'd':
					__output_d(&info, va_arg(ap, int));
					break;
#ifndef INHIBIT_FLOAT
				case 'f':
					__output_f(&info, va_arg(ap, double));
					break;
#endif
				case 's':
					__output_s(&info, va_arg(ap, char *));
					break;
				case 'u':
					__output_u(&info, va_arg(ap, unsigned long));
					break;
				case 'x':
					__output_x(&info, va_arg(ap, unsigned long));
					break;
				default:
					output('%', cookie);
					output(c, cookie);
					c = 0;
					break;
				}
				mode = 0;
			}
			break;
		default:
			if (c == '%') {
				mode = c;
				info.padding = ' ';
				info.maxwidth = -1;
				info.precision = -1;
				info.leftalign = 0;
			} else {
				output(c, cookie);
			}
			break;
		}
	
	}

	return 0;
}

int __format(void (*output)(int c, void *cookie), void *cookie, const char *format, ...)
{
	int re;
	va_list ap;
	va_start(ap, format);
	re = __vformat(output, cookie, format, ap);
	va_end(ap);
	return re;
}


